;;; Verum-Dezyne --- An IDE for Dezyne
;;;
;;; Copyright © 2019,2020,2021,2022,2023 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2019,2020 Rob Wieringa <Rob.Wieringa@verum.com>
;;; Copyright © 2020 Paul Hoogendijk <paul.hoogendijk@verum.com>
;;;
;;; This file is part of Verum-Dezyne.
;;;
;;; Verum-Dezyne is property of Verum Software Tools BV <support@verum.com>.
;;; All rights reserved.

(define-module (transport Websocket)
  #:use-module (ice-9 match)

  #:use-module (dzn runtime)
  #:use-module (ide util)
  #:use-module (ide shell-util)

  #:use-module (oop goops)
  #:use-module (8sync)
  #:use-module (8sync systems websocket client)
  #:use-module (8sync systems websocket server)
  #:use-module (transport transport_interface)
  #:duplicates (merge-generics)
  #:export (<transport:Websocket>
            .clients
            .websocket
            websocket-listen))

(define-class <transport:Websocket> (<dzn:component>)
  (websocket #:accessor .websocket #:init-keyword #:websocket #:init-value #f)

  (clients #:accessor .clients #:init-form (list))
  (wss #:accessor .wss #:init-value #f)
  (disconnect #:accessor .disconnect #:init-value #f)
  (send #:accessor .send #:init-value #f)
  (receive #:accessor .receive #:init-value #f))

(define-method (initialize (o <transport:Websocket>) args)
  (next-method)
  (set! (.websocket o)
        (make <transport:Iwebsocket>
          #:in (make <transport:Iwebsocket.in>
                 #:name "websocket"
                 #:self o
                 #:listen (lambda args (call-in o (lambda _ (apply websocket-listen (cons o args))) `(,(.websocket o) listen)))
                 #:open (lambda args (call-in o (lambda _ (apply websocket-open (cons o args))) `(,(.websocket o) open)))
                 #:connection (lambda args (call-in o (lambda _ (apply websocket-connection (cons o args))) `(,(.websocket o) connection)))
                 #:close (lambda args (call-in o (lambda _ (apply websocket-close (cons o args))) `(,(.websocket o) close)))
                 #:send (lambda args (call-in o (lambda _ (apply websocket-send (cons o args))) `(,(.websocket o) send)))
                 #:get (lambda args (call-in o (lambda _ (apply websocket-get (cons o args))) `(,(.websocket o) get))))
          #:out (make <transport:Iwebsocket.out>))))

(define-method (websocket-listen (o <transport:Websocket>) port)
  (catch #t          ;expect: wrong-type-arg (open port), system-error
    (lambda _
      (let* ((log (dzn:get (.locator o) <procedure> 'log))
             (log-debug (dzn:get (.locator o) <procedure> 'log-debug))
             (verbosity (dzn:get (.locator o) <integer> 'verbosity))
             (path (string-join (dzn:path o) "."))
             (hive (dzn:get (.locator o) <hive>))
             (wss (create-actor
                   hive
                   <websocket-server>
                   #:port port
                   #:on-ws-connection (lambda (ws-id)
                                        (let ((ws ((@@ (8sync actors) hive-resolve-local-actor) hive ws-id)))
                                          (log-debug "on-ws-connection: ws=~s\n" ws)
                                          (when verbosity
                                            (log "connect: ~a\n" path))))
                   #:on-ws-close (lambda (ws)
                                   (log-debug "on-close: ~s\n" ws)
                                   (when verbosity
                                     (log "disconnect: ~a\n" path))
                                   (set! (.clients o) (delete (actor-id ws) (.clients o)))
                                   (when (null? (.clients o))
                                     (action o .websocket .out .disconnect ws "Websocket.scm:disconnect")))
                   #:on-ws-error (lambda (ws e)
                                   (log-debug "on-error: ~s e=~s\n" ws e))
                   #:on-ws-message (lambda (ws msg)
                                     (log-debug "on-message: ~s: ~s\n" ws msg)
                                     (action o .websocket .out .message ws msg))
                   #:on-ws-open (lambda (ws)
                                  (log-debug "on-open: ~s\n" ws)
                                  (let ((clients (.clients o)))
                                    (set! (.clients o) (cons (actor-id ws) clients))
                                    (when (null? clients)
                                      (action o .websocket .out .connected ws)))))))
        (log "Websocket.listen port=~s\n" port)
        (set! (.wss o) wss)
        (action o .websocket .out .listening)))
    (lambda (key . args)
      (unless (and (memq key '(system-error wrong-type-arg))
                   (match args
                     (((or "get-u8" "get-bytevector-n" "lookahead-u8"
                           "put-u8" "put-bytevector") arg ...) #t)
                     (_ #f)))
        (apply throw key args))
      (log-error "error on port ~a: ~a ~s\n" port key args) ;; FIXME: better error message
      (action o .websocket .out .error #f args))))

(define-method (websocket-open (o <transport:Websocket>) uri)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Websocket.websocket-open\n")
    (let* ((hive (dzn:get (.locator o) <hive>))
           (client (create-actor hive <websocket>
                                 #:on-close (lambda (ws)
                                              (log-debug "on-close: ~s\n" ws)
                                              (set! (.clients o) '())
                                              (action o .websocket .out .disconnect ws "msg"))
                                 #:on-error (lambda (ws e)
                                              (log-debug "on-error: ~s e=~s\n" ws e)
                                              (action o .websocket .out .error ws e)
                                              (when (mingw?)
                                                (throw 'error "destroy stale <websocket>" ws)))
                                 #:on-message (lambda (ws msg)
                                                (log-debug "on-message: ~s: ~s\n" ws msg)
                                                (action o .websocket .out .message ws msg))
                                 #:on-open (lambda (ws)
                                             (log-debug "on-open: ~s\n" ws)
                                             (set! (.clients o) (list (actor-id ws)))
                                             (action o .websocket .out .connected ws)))))
      (set! (.clients o) (list client))
      (<- client 'open uri))))

(define-method (websocket-connection (o <transport:Websocket>) ws)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Websocket.websocket-connection\n"))
  (set! (.clients o) (list (actor-id ws)))
  (<- (actor-id ws) 'connected ws))

(define-method (websocket-close (o <transport:Websocket>))
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug)))
    (log-debug "Websocket.websocket-close\n"))
  (for-each (lambda (id) (<- id 'close)) (.clients o)))

(define-method (websocket-send (o <transport:Websocket>) msg)
  (let ((log-debug (dzn:get (.locator o) <procedure> 'log-debug))
        (hive (dzn:get (.locator o) <hive>)))
    (log-debug "Websocket.websocket-send\n")
    (for-each (lambda (id)
                (let ((actor ((@@ (8sync actors) hive-resolve-local-actor) hive id)))
                  (<-wait id 'send msg)))
              (slot-ref o 'clients))))

(define-method (websocket-get (o <transport:Websocket>) ws)
  *unspecified*)
