;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2021, 2022, 2023, 2024, 2025 Janneke Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2025 Rutger van Beusekom <rutger@dezyne.org>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.

(define-module (dzn code scmackerel code)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-9 gnu)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)

  #:use-module (ice-9 match)
  #:use-module (scmackerel code)

  #:use-module (dzn ast ast)
  #:use-module (dzn ast util)
  #:use-module (dzn ast)
  #:use-module (dzn code)
  #:use-module (dzn code language dzn)
  #:use-module (dzn config)
  #:use-module (dzn misc)
  #:export (ast->code
            ast->expression
            code:binding->connect
            code:enum->enum
            code:enum->sm:enum-struct
            code:->formal
            code:caption
            code:generated-comment
            code:member->variable
            code:version-comment
            code:->namespace))

;;;
;;; Helpers.
;;;
(define (code:caption str)
  (let* ((width (floor/ (- 80 (+ 2 (string-length str))) 2))
         (pad (make-string width #\/)))
    (string-append pad " " str " " pad)))

(define (code:generated-comment o)
  (sm:comment* (simple-format #f "// Generated by dzn code from ~a\n"
                              (ast:source-file o))))

(define (code:version-comment)
  (sm:comment* (simple-format #f "// version ~a\n" %package-version)))

(define-method (code:enum->enum (o <enum>))
  (sm:enum
   (name (ast:name o))
   (fields (map (cute string-append name "_" <>) (ast:field* o)))))

(define-method (code:enum->sm:enum-struct (o <enum>))
  (sm:enum-struct
   (name (ast:name o))
   (fields (ast:field* o))))

(define-method (code:->namespace (o <list>) code)
  (let* ((code (fold (lambda (space code)
                       (sm:namespace* space code))
                     code
                     (reverse o))))
    (if (pair? code) code
        (list code))))

(define-method (code:->namespace (o <ast>) code)
  (code:->namespace (ast:full-scope o) code))

(define-method (code:->namespace (o <enum>) code)
  (let* ((model (ast:parent o <model>))
         (scope (if model model o)))
    (code:->namespace (ast:full-scope scope) code)))

(define-method (code:member->variable (o <variable>))
  (sm:variable
   (type (code:type-name o))
   (name (.name o))
   (expression (ast->expression (.expression o)))))

(define-method (code:->formal (o <formal>))
  (let ((type (code:type-name o)))
    (sm:formal (type type)
               (name (.name o)))))

(define* (code:binding->connect binding #:key (name "connect"))
  (let* ((provides
          requires
          (code:provides+requires-end-point binding)))
    (sm:call (name name)
             (arguments
              (list
               (simple-format
                #f "~a~a"
                (%member-prefix) (code:end-point->string provides))
               (simple-format
                #f "~a~a"
                (%member-prefix) (code:end-point->string requires)))))))


;;;
;;; Ast->code.
;;;
(define-method (ast->code (o <top>))
  o)

(define-method (ast->code (o <object>))
  (throw 'error "ast->code specialization missing for" o))

(define-method (ast->code (o <guard>))
  (let ((expression (.expression o))
        (statement (.statement o)))
    (cond
     ((is-a? expression <otherwise>)
      (ast->code statement))
     ((ast:literal-true? expression)
      (cond
       ((and (is-a? statement <compound>)
             ;; XXX TODO
             ;;(as (c++:statement* statement) <pair>)
             (as (ast:statement* statement) <pair>))
        =>
        (lambda (statements)
          (sm:statements* (map ast->code statements))))
       ((is-a? statement <compound>)
        (sm:statement*))
       (else
        (ast->code statement))))
     (else
      (sm:if* (ast->expression expression)
              (ast->code statement))))))

(define-method (declarative-compound->code (o <compound>))
  (define (guard->code guard code)
    (cond ((and (sm:if? guard)
                (not (sm:if-else guard)))
           (set-field guard (sm:if-else) code))
          ((sm:if? code)
           (set-field code (sm:if-else) guard))
          (code
           (throw 'programming-error "have code and guard"))
          (else
           guard)))
  (let* ((statements (ast:statement* o))
         (statements (map ast->code statements)))
    (sm:compound*
     (fold guard->code #f (reverse statements)))))

(define-method (ast->code (o <declarative-compound>))
  (declarative-compound->code o))

;;; imperative
(define-method (ast->code (o <compound>))
  (let ((statements (ast:statement* o))) ;; XXX FIXME: transformation
    (match statements
      ((statement)
       (let ((statement (ast->code statement)))
         (if (sm:compound? statement) statement
             (sm:compound* statement))))
      ((? (const (ast:declarative? o)))
       (declarative-compound->code o))
      (_
       (sm:compound* (map ast->code statements))))))

(define-method (ast->code (o <skip>))
  (sm:statement*))

(define-method (ast->code (o <action>))
  (let* ((action-name (code:event-name o))
         (arguments (ast:argument* o))
         (arguments (map ast->expression arguments))
         (action (sm:call (name (string-append (%member-prefix) action-name))
                          (arguments arguments))))
    action))

(define-method (ast->code (o <assign>))
  (let* ((var (ast->expression (.variable o)))
         (expression (.expression o))
         (action? (is-a? expression <action>))
         (expression (if (not action?) (ast->expression expression)
                         (ast->code expression))))
    (sm:assign* var expression)))

(define-method (ast->code (o <variable>))
  (let ((expression (.expression o)))
    (cond
     ((not (is-a? expression <action>))
      (sm:variable
       (type (code:type-name o))
       (name (.name o))
       (expression (ast->expression expression))))
     (else
      (let ((variable
             (sm:variable
              (type (code:type-name o))
              (name (.name o))
              (expression (ast->code expression)))))
        variable)))))

(define (ast->namespace ast)
  (let loop ((ast ast))
    (if (is-a? ast <root>) '()
        (append (loop (ast:parent ast <namespace>))
                (list (ast:name ast))))))

(define-method (ast->code (o <call>))
  (let* ((function (.function o))
         (name (string-join (ast->namespace function) (%type-infix)))
         (arguments (map ast->expression (ast:argument* o))))
    (if (ast:parent function <behavior>)
        (sm:call-method
         (name name)
         (arguments arguments))
        (sm:call
         (name name)
         (arguments arguments)))))

(define-method (ast->code (o <reply>))
  (let ((p (.parent o)))
    (cond
     ((or (is-a? p <guard>) (is-a? p <if>))
      (ast->code (code:wrap-compound o)))
     (else
      (let ((type (ast:type o)))
        (if (is-a? type <void>) (sm:statement*)
            (sm:assign* (sm:member* (%member-prefix) (code:reply-var type))
                        (ast->expression (.expression o)))))))))

(define-method (ast->code (o <declarative-illegal>))
  (ast->code (make <illegal>)))

(define-method (ast->code (o <if>))
  (sm:if* (ast->expression (.expression o))
          (ast->code (.then o))
          (and=> (.else o) ast->code)))

(define-method (ast->code (o <return>))
  (sm:return* (ast->expression (.expression o))))


;;;
;;; Ast->expression  -- Common ast to expression tranformations.
;;;

;;; TODO: Deal with language overrides and code dupliciation, especially
;;; in the face of recursion.  Current solution introduces a language
;;; prefixed override for ast->expression, e.g., makreel:ast->expression
;;; and specializes for <shared-*>.  However this also requires copying
;;; and prefixing recursive version of ast->expression, e.g., <binary>,
;;; <not>, <group>.

(define-method (ast->expression (o <top>))
  o)

(define-method (ast->expression (o <action>))
  (ast->code o))

(define-method (ast->expression (o <call>))
  (ast->code o))

(define-method (ast->expression (o <binary>))
  (sm:expression
   (operator (operator->string o))
   (operands (list (ast->expression (.left o))
                   (ast->expression (.right o))))))

(define-method (ast->expression (o <group>))
  (sm:group* (ast->expression (.expression o))))

(define-method (ast->expression (o <literal>))
  (sm:literal
   (value (.value o))))

(define-method (ast->expression (o <not>))
  (sm:not* (ast->expression (.expression o))))

(define-method (ast->expression (o <type>))
  (code:type-name o))

(define-method (ast->expression (o <data>))
  (.value o))

(define-method (ast->expression (o <void>))
  #f)

(define-method (ast->expression (o <literal>))
  (match (.value o)
    ("void" #f)
    (value (ast->expression value))))

(define-method (ast->expression (o <enum-literal>))
  (let ((lst (append (ast:full-name (.type o)) (list (.field o)))))
    (string-append (%type-prefix)
                   (string-join lst (%type-infix)))))

(define-method (ast->expression (o <otherwise>))
  (ast->code (make <illegal>)))

(define-method (ast->expression (o <var>))
  (let ((name (.name o)))
    (if (ast:member? (.variable o)) (sm:member* (%member-prefix) name)
        name)))

(define-method (ast->expression (o <variable>))
  (let ((name (.name o)))
    (if (ast:member? o) (sm:member* (%member-prefix) name)
        name)))

(define-method (ast->expression (o <formal>))
  (.name o))

(define-method (ast->expression (o <field-test>))
  (let* ((enum-literal (make <enum-literal>
                         #:type.name (.type.name (.variable o))
                         #:field (.field o)))
         (var (make <var> #:name (.variable.name o)))
         (expression (graft o (make <equal> #:left var #:right enum-literal))))
    (ast->expression expression)))

(define-method (ast->expression (o <shared-field-test>))
  (let* ((variable (.variable o))
         (type (.type variable))
         (type-name (make <scope.name>
                      #:ids (ast:full-name type)))
         (enum-literal (make <enum-literal>
                         #:type.name type-name
                         #:field (.field o)))
         (name (.name variable))
         (port-name (.port.name o))
         (var (make <shared-var>
                #:name name
                #:port.name port-name))
         (expression (graft o (make <equal>
                                #:left var
                                #:right enum-literal))))
    (ast->expression expression)))

(define-method (ast->expression (o <shared-var>))
  (let ((lst (ast:full-name o)))
    (string-join lst (%name-infix))))

(define-method (ast->expression (o <shared-variable>))
  (let* ((name (ast:full-name o))
         (name (string-join name (%name-infix))))
    (if (ast:member? o) (sm:member* (%member-prefix) name)
        name)))
